<?php

namespace App\Http\Controllers;

use App\Models\Contact;
use App\Models\School;
use App\Models\Position;
use App\Models\Year;
use App\Models\Enrollment;
use App\Models\Section;
use App\Models\Student;
use App\Models\Teacher;
use App\Mail\ContactChanged;
use Illuminate\Support\Facades\Mail;
use App\Http\Requests\StoreContactRequest;
use App\Http\Requests\UpdateContactRequest;
use App\Traits\ContactPermissionTrait;
use Carbon\Carbon;

class ContactController extends Controller
{
    use ContactPermissionTrait;

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        if ($this->checkContactPermissions('index', 0) == false) {
            return redirect(route('dashboard'))->with('error', 'You do not have permission to view contacts.');
        }

        if (auth()->user()->role_id == 3) {
            $contacts = Contact::where('school_id', auth()->user()->school_id)->orderBy('last_name')->get();
        } else {
            $contacts = Contact::all()->sortBy('last_name');
        }

        return view('admin.contacts.index')
            ->with('title', 'Contacts')
            ->with('contacts', $contacts);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        if ($this->checkContactPermissions('create', 0) == false) {
            return redirect(route('contacts.index'))->with('error', 'You do not have permission to create contacts.');
        }

        if (auth()->user()->role_id == 3) {
            $schools = School::where('id', auth()->user()->school_id)->get();
        } else {
            $schools = School::orderBy('name')->get();
        }
        return view('admin.contacts.form')
            ->with('action', 'create')
            ->with('title', 'Create Contact')
            ->with('positions', Position::orderBy('position')->get())
            ->with('schools', $schools);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreContactRequest $request)
    {
        if ($this->checkContactPermissions('store', 0) == false) {
            return redirect(route('contacts.index'))->with('error', 'You do not have permission to store that contact.');
        }
        $contact = Contact::create($request->all());
        $this->sendContactNotification($contact, 'Added');
        return redirect(route('contacts.index'))->with('success', 'Contact created successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(Contact $contact)
    {
        if ($this->checkContactPermissions('show', $contact->id) == false) {
            return redirect(route('contacts.index'))->with('error', 'You do not have permission to view that contact.');
        }
        return view('admin.contacts.show', ['contact' => $contact])
            ->with('title', 'View ' . $contact->first_name . ' ' . $contact->last_name);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Contact $contact)
    {
        if ($this->checkContactPermissions('edit', $contact->id) == false) {
            return redirect(route('contacts.index'))->with('error', 'You do not have permission to edit that contact.');
        }
        return view('admin.contacts.form')
            ->with('action', 'edit')
            ->with('title', 'Edit ' . $contact->first_name . ' ' . $contact->last_name)
            ->with('contact', $contact)
            ->with('positions', Position::orderBy('position')->get())
            ->with('schools', School::orderBy('name')->get());
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateContactRequest $request, Contact $contact)
    {
        if ($this->checkContactPermissions('update', $contact->id) == false) {
            return redirect(route('contacts.index'))->with('error', 'You do not have permission to update that contact.');
        }
        $contact->update($request->all());
        $this->sendContactNotification($contact, 'Changed');
        return redirect(route('contacts.index'))->with('success', 'Contact ' . $contact->first_name . ' ' . $contact->last_name . ' updated successfully!');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function delete(Contact $contact)
    {
        if ($this->checkContactPermissions('delete', $contact->id) == false) {
            return redirect(route('contacts.index'))->with('error', 'You do not have permission to delete that contact.');
        }
        return view('admin.contacts.delete')
            ->with('contact', $contact)
            ->with('title', 'Delete ' . $contact->first_name . ' ' . $contact->last_name);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Contact $contact)
    {
        if ($this->checkContactPermissions('destroy', $contact->id) == false) {
            return redirect(route('contacts.index'))->with('error', 'You do not have permission to destroy that contact.');
        }
        $this->sendContactNotification($contact, 'Deleted');
        $contact->delete();
        return redirect(route('contacts.index'))->with('success', 'Contact ' . $contact->first_name . ' ' . $contact->last_name . ' deleted successfully!');
    }

    /**
     * Send email notifications to teachers about contact changes.
     */
    protected function sendContactNotification(Contact $contact, string $action)
    {
        // Find the current academic year
        $currentDate = Carbon::today();
        $year = Year::where('start_date', '<=', $currentDate)
            ->where('end_date', '>=', $currentDate)
            ->first();

        if (!$year) {
            return; // Skip if no current year found
        }

        // Find active enrollments for the contact's school in the current year
        $enrollments = Enrollment::where('year_id', $year->id)
            ->where('status', 'A')
            ->whereHas('student', function ($query) use ($contact) {
                $query->where('school_id', $contact->school_id);
            })
            ->with(['section', 'student', 'section.teacher'])
            ->get();

        if ($enrollments->isEmpty()) {
            return; // Skip if no enrollments found
        }

        // Group enrollments by teacher
        $teacherSections = [];
        foreach ($enrollments as $enrollment) {
            $section = $enrollment->section;
            if (!$section || !$section->teacher) {
                continue; // Skip if no section or teacher
            }
            $teacherId = $section->teacher_id;
            if (!isset($teacherSections[$teacherId])) {
                $teacherSections[$teacherId] = [
                    'teacher' => $section->teacher,
                    'sections' => [],
                ];
            }
            if (!isset($teacherSections[$teacherId]['sections'][$section->id])) {
                $teacherSections[$teacherId]['sections'][$section->id] = [
                    'name' => $section->name,
                    'students' => [],
                ];
            }
            $teacherSections[$teacherId]['sections'][$section->id]['students'][] = $enrollment->student;
        }

        // Send emails to each teacher
        foreach ($teacherSections as $teacherData) {
            $teacher = $teacherData['teacher'];
            if (!$teacher->email) {
                continue; // Skip if no email
            }
            $sections = array_values($teacherData['sections']);
            $schoolName = $contact->school->name;
            Mail::to($teacher->email)->send(new ContactChanged($contact, $action, $sections, $schoolName));
        }
    }
}