<?php

namespace App\Http\Controllers;

use App\Models\Enrollment;
use App\Models\Section;
use App\Models\Student;
use App\Models\Contact;
use App\Models\Year;
use App\Models\School;
use App\Models\Teacher;
use App\Models\Provider;
use App\Models\WelcomeLetter;
use App\Http\Requests\StoreEnrollmentRequest;
use App\Http\Requests\UpdateEnrollmentRequest;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use App\Services\EmailService;
use Illuminate\Http\Request;
use App\Traits\EnrollmentPermissionTrait;
use Carbon\Carbon;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\MessageBag;

class EnrollmentController extends Controller
{
    use EnrollmentPermissionTrait;
    protected $emailService;

    public function __construct(EmailService $emailService)
    {
        $this->emailService = $emailService;
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        if (auth()->user()->role_id == 4) {
            $yearId = Year::where('is_visible_to_teachers', true)->orderBy('year', 'desc')->first()->id;
        } else {
            $yearId = Year::orderBy('year', 'desc')->first()->id;
        }

        return $this->index2($yearId, 0,0,0,0,0, 'A', 'A', 0, 'ALL', 'index');
    }

    public function waitList()
    {
        if (auth()->user()->role_id == 4) {
            return redirect(route('dashboard'))->with('error','You do not have permission to view this.');
        }
        return $this->index2(Year::orderBy('year', 'desc')->first()->id, 0,0,0,0,0, 'W', 'A', 0, 'ALL', 'waitlist');
    }

    public function index2($yearId, $schoolId, $providerId, $departmentId, $sectionId, $teacherId, $status, $term, $grade, $dualCredit, $page)
    {
        $title = ($page == 'index') ? 'Enrollments' : 'Waitlisted Enrollments';

        $query = Section::with(['provider', 'course', 'teacher'])
            ->where('year_id', $yearId)
            ->where('is_cancelled', false);

        switch(auth()->user()->role_id) {
            case 2:
                $schools = School::orderBy('name', 'asc')->get();
                $teachers = Teacher::orderBy('last_name', 'asc')->get();
                $providers = Provider::where('id', auth()->user()->provider_id)->get();
                $years = Year::orderBy('year', 'desc')->get();
                $providerId = auth()->user()->provider_id;
                $query->where('provider_id', $providerId);
                break;
            case 3:
                $schools = School::where('id', auth()->user()->school_id)->get();
                $teachers = Teacher::orderBy('last_name', 'asc')->get();
                $providers = Provider::orderBy('name', 'asc')->get();
                $years = Year::orderBy('year', 'desc')->get();
                $schoolId = auth()->user()->school_id;
                break;
            case 4:
                $schools = School::orderBy('name', 'asc')->get();
                $teachers = Teacher::where('id', auth()->user()->teacher_id)->get();
                $providers = Provider::orderBy('name', 'asc')->get();
                $years = Year::where('is_visible_to_teachers', true)->orderBy('year', 'desc')->get();
                $query->where('teacher_id', auth()->user()->teacher_id);
                $teacherId = auth()->user()->teacher_id;
                break;
            default:
                $schools = School::orderBy('name', 'asc')->get();
                $teachers = Teacher::orderBy('last_name', 'asc')->get();
                $providers = Provider::orderBy('name', 'asc')->get();
                $years = Year::orderBy('year', 'desc')->get();
        }

        $sections = $query->get()->sortBy('course.name');

        $enrollments = Enrollment::enrollmentSQLBuilder($yearId, $schoolId, $providerId, $departmentId, $sectionId, $teacherId, $status, $term, $grade, $dualCredit);

        return view('admin.enrollments.index', compact(
            'yearId', 'schoolId', 'providerId', 'sectionId', 'teacherId', 'status', 'term', 'grade', 'dualCredit',
            'schools', 'teachers', 'providers', 'sections', 'title', 'page', 'enrollments'
        ))
        ->with('years', $years);
    }

    public function filterEnrollments(Request $request)
    {
        return $this->index2($request->year_id, $request->school_id, $request->provider_id, $request->department_id, $request->section_id, $request->teacher_id, $request->status, $request->term, $request->grade, $request->dual_credit, 'index');
    }

    public function filterWaitlistEnrollments(Request $request)
    {
        return $this->index2($request->year_id, $request->school_id, $request->provider_id, $request->department_id, $request->section_id, $request->teacher_id, 'W', $request->term, $request->grade, $request->dual_credit, 'waitlist');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        if($this->checkEnrollmentPermissions('create', NULL) == false){
            return redirect(route('dashboard'))->with('error','You do not have permission to create enrollments.');
        }
        return $this->create2(Year::orderBy('year', 'desc')->first()->id, 'A');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create2($yearid, $term)
    {
        $userRole = auth()->user()->role_id;
        $schoolId = $userRole == 3 ? auth()->user()->school_id : null;

        $studentQuery = Student::where('grade', '<', 13)->orderBy('last_name');

        if ($userRole == 3) {
            $studentQuery->where('school_id', $schoolId);
        }

        $students = $studentQuery->get();

        $sectionQuery = Section::where('year_id', $yearid)
            ->where('is_cancelled', false);

        if ($term != 'A') {
            $sectionQuery->where('term', $term);
        }

        if ($userRole == 3) {
            $sectionQuery->where(function ($q) use ($schoolId) {
                $q->whereNull('restricted_school_ids')
                  ->orWhereJsonContains('restricted_school_ids', (int) $schoolId);
            });
        }

        $sections = $sectionQuery->get()->sortBy(['provider.name', 'course.name']);

        return view('admin.enrollments.form')
            ->with('action', 'create')
            ->with('title', 'Create Enrollment')
            ->with('sections', $sections)
            ->with('section', $sections->first())
            ->with('schools', School::orderBy('name')->get())
            ->with('students', $students)
            ->with('providers', Provider::orderBy('name')->get())
            ->with('years', Year::orderBy('year', 'desc')->get()->take(1))
            ->with('term', $term)
            ->with('yearid', $yearid);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreEnrollmentRequest $request)
    {
        if ($this->checkEnrollmentPermissions('store', null) == false) {
            return redirect(route('dashboard'))->with('error', 'You do not have permission to store enrollments.');
        }

        $student = Student::find($request->student_id);
        $section = Section::find($request->section_id);

        if (!$section->isSchoolAllowed($student->school_id)) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Student is not from an allowed school for this section.');
        }

        $nextYearGrade = $student->grade + 1;

        if ($section->minimum_grade && $nextYearGrade < $section->minimum_grade) {
            return redirect()->back()
                ->withInput()
                ->with('error', "Student's grade for next year ($nextYearGrade) does not meet the minimum grade requirement of {$section->minimum_grade} for this section.");
        }

        $enrollment = new Enrollment;
        $enrollment->year_id = $request->year_id;
        $enrollment->student_id = $request->student_id;
        $enrollment->section_id = $request->section_id;
        $enrollment->course_id = $section->course_id;

        if ($section->dual_credit == true) {
            $enrollment->dual_credit = $request->dual_credit;
        } else {
            $enrollment->dual_credit = false;
        }

        if (auth()->user()->role_id == 1) {
            if ($request->status) {
                $enrollment->status = $request->status;
            } else {
                $enrollment->status = $section->openSeats() > 0 ? 'A' : 'W';
            }
            $enrollment->lms_updated = false;
            $enrollment->ps_updated = false;
        } else {
            if ($section->openSeats() > 0) {
                $enrollment->status = 'A';
            } else {
                $enrollment->status = 'W';
            }
            $enrollment->lms_updated = false;
            $enrollment->ps_updated = false;
        }

        $enrollment->save();

        return redirect(route('enrollments.index'))->with('success', 'Enrollment created successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(Enrollment $enrollment)
    {
        if($this->checkEnrollmentPermissions('show', $enrollment->id) == false){
            return redirect(route('enrollments.index'))->with('error','You do not have permission to see that enrollment.');
        }

        $facilitators = $this->getFacilitatorsForEnrollment($enrollment);

        return view('admin.enrollments.show', [
            'enrollment' => $enrollment,
            'facilitators' => $facilitators,
            'title' => 'View ' . $enrollment->student->first_name . ' ' . $enrollment->student->last_name
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Enrollment $enrollment)
    {
        if($this->checkEnrollmentPermissions('edit', $enrollment->id) == false){
            return redirect(route('enrollments.index'))->with('error','You do not have permission to edit that enrollment.');
        }

        $userRole = auth()->user()->role_id;
        $latestYear = Year::orderBy('year', 'desc')->first();

        if (($userRole != 1 && $userRole != 2) && $enrollment->year_id != $latestYear->id) {
            return redirect(route('enrollments.index'))->with('error', 'You cannot edit this enrollment because it is not for the latest year.');
        }

        if ($userRole == 1 || $userRole == 2) {
            $years = Year::orderBy('year', 'desc')->get();
        } else {
            $years = Year::orderBy('year', 'desc')->get()->take(1);
        }

        $term = $enrollment->section->term ?? 'A';

        return view('admin.enrollments.form')
            ->with('action', 'edit')
            ->with('title', 'Edit Enrollment')
            ->with('enrollment', $enrollment)
            ->with('sections', Section::where('year_id', $latestYear->id)->get()->sortBy(['provider.name', 'course.name']))
            ->with('section', $enrollment->section)
            ->with('schools', School::orderBy('name')->get())
            ->with('students', Student::orderby('last_name')->get())
            ->with('providers', Provider::orderBy('name')->get())
            ->with('years', $years)
            ->with('yearid', $enrollment->year_id)
            ->with('term', $term);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateEnrollmentRequest $request, Enrollment $enrollment)
    {
        if (auth()->user()->role_id == 4) {
            return redirect(route('dashboard'))->with('error', 'You do not have permission to view this.');
        }
        if ($this->checkEnrollmentPermissions('update', $enrollment->id) == false) {
            return redirect(route('enrollments.index'))->with('error', 'You do not have permission to update that enrollment.');
        }

        $student = Student::find($request->student_id);
        $section = Section::find($request->section_id);

        if (!$section->isSchoolAllowed($student->school_id)) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Student is not from an allowed school for this section.');
        }

        $nextYearGrade = $student->grade + 1;

        if ($section->minimum_grade && $nextYearGrade < $section->minimum_grade) {
            return redirect()->back()
                ->withInput()
                ->with('error', "Student's grade for next year ($nextYearGrade) does not meet the minimum grade requirement of {$section->minimum_grade} for this section.");
        }

        $enrollment->year_id = $request->year_id;
        $enrollment->student_id = $request->student_id;
        $enrollment->section_id = $request->section_id;
        $enrollment->course_id = $section->course_id;
        $enrollment->dual_credit = $request->dual_credit;

        if (auth()->user()->role_id == 1) {
            $enrollment->status = $request->status;
            $enrollment->lms_updated = $request->lms_updated;
            $enrollment->ps_updated = $request->ps_updated;
        } else {
            if ($section->openSeats() > 0) {
                $enrollment->status = 'A';
            } else {
                $enrollment->status = 'W';
            }
            $enrollment->lms_updated = false;
            $enrollment->ps_updated = false;
        }

        if ($request->prev_status != $request->status) {
            $enrollment->lms_updated = false;
            $enrollment->ps_updated = false;
        }

        $enrollment->update();
        return redirect(route('enrollments.index'))->with('success', 'Enrollment updated successfully!');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function delete(Enrollment $enrollment)
    {
        if($this->checkEnrollmentPermissions('delete', $enrollment->id) == false){
            return redirect(route('enrollments.index'))->with('error','You do not have permission to delete that enrollment.');
        }
        return view('admin.enrollments.delete')
            ->with('enrollment', $enrollment)
            ->with('title', 'Delete Enrollment');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Enrollment $enrollment)
    {
        if($this->checkEnrollmentPermissions('destroy', $enrollment->id) == false){
            return redirect(route('enrollments.index'))->with('error','You do not have permission to destroy that enrollment.');
        }
        $enrollment->delete();
        return redirect(route('enrollments.index'))->with('success','Enrollment deleted successfully!');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function drop(Enrollment $enrollment)
    {
        if($this->checkEnrollmentPermissions('drop', $enrollment->id) == false){
            return redirect(route('enrollments.index'))->with('error','You do not have permission to drop that enrollment.');
        }
        return view('admin.enrollments.drop')->with('enrollment', $enrollment)->with('title', 'Drop Enrollment');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function dropConfirm(Enrollment $enrollment)
    {
        $enrollment->status = 'R';
        $enrollment->lms_updated = false;
        $enrollment->update();
        return redirect(route('enrollments.index'))->with('success','Enrollment dropped successfully!');
    }

    /**
     * When the user selects a section, show/hide the dual credit option depending on
     */
    public function showHideDualCredit($sectionId)
    {
        return view('admin.enrollments.show-hide-dual-credit')->with('section', Section::find($sectionId));
    }

    public function uploadMoodleStudents(Request $request)
    {
        if (auth()->user()->id != 1) {
            return redirect(route('dashboard'))->with('error','You do not have permission to view this.');
        }

        $enrollments = Enrollment::whereHas('section', function($query) {
            $query->where('is_moodle', true);
        })
        ->where('status', '<>', 'W')
        ->where('lms_updated', false)
        ->with(['section' => function($query) {
            $query->select('id', 'is_moodle', 'moodle_short_name');
        }])
        ->get();

        return view('admin.enrollments.upload-moodle-students')
        ->with('title', 'Upload Moodle Students')
        ->with('enrollments', $enrollments);
    }

    public function updateLmsForEnrollments(Request $request)
    {
        $enrollmentIds = $request->input('enrollment_ids');

        if (!empty($enrollmentIds)) {
            Enrollment::whereIn('id', $enrollmentIds)->update(['lms_updated' => true]);
        }

        return redirect()->back()->with('success', 'Selected enrollments have been updated.');
    }

    public function uploadPowerSchoolStudents(Request $request)
    {
        if (auth()->user()->role_id != 1) {
            return redirect(route('dashboard'))->with('error','You do not have permission to do this.');
        }

        $enrollments = Enrollment::where('ps_updated', false)->get();

        return view('admin.enrollments.upload-powerschool-students')
        ->with('title', 'Upload PowerSchool Students')
        ->with('enrollments', $enrollments);
    }

    public function updatePowerSchoolForEnrollments(Request $request)
    {
        $enrollmentIds = $request->input('enrollment_ids');

        if (!empty($enrollmentIds)) {
            Enrollment::whereIn('id', $enrollmentIds)->update(['ps_updated' => true]);
        }

        return redirect()->back()->with('success', 'Selected enrollments have been updated in PowerSchool.');
    }

    private function getFacilitatorsForEnrollment($enrollment)
    {
        return Contact::where('is_facilitator', true)
            ->where('school_id', $enrollment->student->school_id)
            ->select('contacts.*',
                DB::raw('(SELECT COUNT(*) FROM enrollments e
                          JOIN students s ON e.student_id = s.id
                          WHERE e.section_id = ' . $enrollment->section_id . '
                          AND s.school_id = contacts.school_id
                          AND e.sent_facilitator_copy = 0) as pending_count'))
            ->get();
    }

    private function sendFacilitatorCopy($enrollment)
    {
        $facilitators = Contact::where('school_id', $enrollment->student->school_id)
            ->where('is_facilitator', true)
            ->get();

        $existingSentCopy = Enrollment::where('section_id', $enrollment->section_id)
            ->where('sent_facilitator_copy', true)
            ->exists();

        if (!$existingSentCopy && $facilitators->isNotEmpty()) {
            foreach ($facilitators as $facilitator) {
                $this->emailService->sendWelcomeEmail(
                    $enrollment->section_id,
                    $facilitator->email,
                    $enrollment->section->teacher->email,
                    $enrollment->section->teacher->first_name . " " . $enrollment->section->teacher->last_name,
                    'Facilitator Copy: Welcome to ' . $enrollment->section->course->name,
                    WelcomeLetter::where('section_id', $enrollment->section_id)->first()->text,
                    NULL
                );
            }

            Enrollment::where('section_id', $enrollment->section_id)
                ->whereHas('student', function ($query) use ($enrollment) {
                    $query->where('school_id', $enrollment->student->school_id);
                })
                ->update(['sent_facilitator_copy' => true]);
        } else {
            $enrollment->sent_facilitator_copy = true;
        }
    }

    private function sendWelcomeLetter(Enrollment $enrollment)
    {
        $welcomeLetter = WelcomeLetter::where('section_id', $enrollment->section_id)->first();

        if ($welcomeLetter) {
            $this->emailService->sendWelcomeEmail(
                $enrollment->section_id,
                $enrollment->student->email,
                $enrollment->section->teacher->email,
                $enrollment->section->teacher->first_name . " " . $enrollment->section->teacher->last_name,
                'Welcome to ' . $enrollment->section->course->name,
                $welcomeLetter->text,
                $enrollment->student->first_name . " " . $enrollment->student->last_name
            );
        }
    }

    /**
     * Fetch students eligible for a section based on restricted_school_ids.
     */
    public function studentsForSection($sectionId)
    {
        $section = Section::findOrFail($sectionId);
        $query = Student::where('grade', '<', 13)->orderBy('last_name');

        if (!empty($section->restricted_school_ids)) {
            $query->whereIn('school_id', $section->restricted_school_ids);
        }

        $students = $query->get(['id', 'first_name', 'last_name']);

        return response()->json(['students' => $students]);
    }



public function BPSMedTermStudentUploadPreview(Request $request)
{
    if (auth()->user()->id != 1) {
        return redirect()->route('dashboard')->with('error', 'You do not have permission to view this.');
    }

    $request->validate([
        'csv_file' => 'required|file|mimes:csv,txt',
        'fall_bhs' => 'nullable|exists:sections,id',
        'spring_bhs' => 'nullable|exists:sections,id',
        'fall_chs' => 'nullable|exists:sections,id',
        'spring_chs' => 'nullable|exists:sections,id',
        'fall_lhs' => 'nullable|exists:sections,id',
        'spring_lhs' => 'nullable|exists:sections,id',
    ]);

    $yearId = Year::orderBy('year', 'desc')->first()->id;
    $year = Year::find($yearId)->year;
    $title = 'BPS Medical Terminology Upload Preview';

    $data = Excel::toArray([], $request->file('csv_file'))[0];
    $headers = array_shift($data);

    $newStudents = [];
    $existingStudents = [];
    $errors = new MessageBag();

    $schoolMap = [
        5 => 'CHS',
        6 => 'BHS',
        8 => 'LHS',
    ];

    // Store selected section IDs from the form for easier lookup
    $selectedSections = [
        'BHS_F' => $request->input('fall_bhs'),
        'BHS_S' => $request->input('spring_bhs'),
        'CHS_F' => $request->input('fall_chs'),
        'CHS_S' => $request->input('spring_chs'),
        'LHS_F' => $request->input('fall_lhs'),
        'LHS_S' => $request->input('spring_lhs'),
    ];

    // Fetch all relevant sections once to avoid repeated DB queries in the loop
    $allSections = Section::whereIn('id', array_filter(array_values($selectedSections)))
                          ->with(['course', 'teacher'])
                          ->get()
                          ->keyBy('id');

    foreach ($data as $row) {
        $rowData = array_combine($headers, $row);
        $rowData = array_map('trim', $rowData);
        $rowData['ssid'] = (int)$rowData['ssid'];
        $rowData['phone'] = $rowData['phone'] ? (string)$rowData['phone'] : null;

        $validator = Validator::make($rowData, [
            'first_name' => 'required|string|max:50',
            'last_name' => 'required|string|max:50',
            'school_id' => 'required|integer|in:5,6,8',
            'email' => 'required|email|max:75',
            'dob' => 'required|date',
            'phone' => 'nullable|string|max:10',
            'grade' => 'required|integer|between:9,12',
            'gender' => 'required|in:M,F',
            'ssid' => 'required|integer',
            'iep' => 'required|boolean',
            'has_504' => 'required|boolean',
            'english_learner' => 'required|boolean',
            'term' => 'required|in:F,S',
        ]);

        if ($validator->fails()) {
            $errors->add('csv_error', "Error for {$rowData['first_name']} {$rowData['last_name']}: " . implode(', ', $validator->errors()->all()));
            continue;
        }

        $schoolId = (int)$rowData['school_id'];
        $term = $rowData['term'];
        $matchingSection = null;
        $sectionIdFromRequest = null;

        // Determine which section_id from the request to use
        if ($schoolId == 6 && $term == 'F') { // BHS Fall
            $sectionIdFromRequest = $selectedSections['BHS_F'];
        } elseif ($schoolId == 6 && $term == 'S') { // BHS Spring
            $sectionIdFromRequest = $selectedSections['BHS_S'];
        } elseif ($schoolId == 5 && $term == 'F') { // CHS Fall
            $sectionIdFromRequest = $selectedSections['CHS_F'];
        } elseif ($schoolId == 5 && $term == 'S') { // CHS Spring
            $sectionIdFromRequest = $selectedSections['CHS_S'];
        } elseif ($schoolId == 8 && $term == 'F') { // LHS Fall
            $sectionIdFromRequest = $selectedSections['LHS_F'];
        } elseif ($schoolId == 8 && $term == 'S') { // LHS Spring
            $sectionIdFromRequest = $selectedSections['LHS_S'];
        }

        if ($sectionIdFromRequest && isset($allSections[$sectionIdFromRequest])) {
            $matchingSection = $allSections[$sectionIdFromRequest];
        }

        if (!$matchingSection) {
            $schoolCode = $schoolMap[$schoolId] ?? 'Unknown';
            $errors->add('csv_error', "No matching section selected in the form for {$rowData['first_name']} {$rowData['last_name']} (School: {$schoolCode}, Term: {$term}). Please ensure a section is selected for this combination.");
            continue;
        }

        $nextYearGrade = (int)$rowData['grade'] + 1;
        if ($matchingSection->minimum_grade && $nextYearGrade < $matchingSection->minimum_grade) {
            $errors->add('csv_error', "Grade error for {$rowData['first_name']} {$rowData['last_name']}: Next year's grade ($nextYearGrade) is below minimum ({$matchingSection->minimum_grade}) for section '{$matchingSection->course->name}'.");
            continue;
        }

        $student = Student::where('email', $rowData['email'])->orWhere('ssid', $rowData['ssid'])->first();
        $sectionName = $matchingSection->course->name . ' (' . $year . ' ' . ($matchingSection->term == 'F' ? 'Fall' : 'Spring') . ') - ' . $matchingSection->teacher->first_name . ' ' . $matchingSection->teacher->last_name;
        $schoolName = $schoolMap[$schoolId] ?? 'Unknown';

        if ($student) {
            $existingStudents[] = [
                'name' => "{$rowData['first_name']} {$rowData['last_name']}",
                'school' => $schoolName,
                'section' => $sectionName,
                'row_data' => $rowData,
                'section_id' => $matchingSection->id,
            ];
        } else {
            $newStudents[] = [
                'name' => "{$rowData['first_name']} {$rowData['last_name']}",
                'school' => $schoolName,
                'section' => $sectionName,
                'row_data' => $rowData,
                'section_id' => $matchingSection->id,
            ];
        }
    }

    return view('admin.enrollments.bps-medterm-upload-preview', compact('newStudents', 'existingStudents', 'errors', 'yearId', 'title'));
}

public function BPSMedTermStudentUpload(Request $request)
{
    if (auth()->user()->id != 1) {
        return redirect()->route('dashboard')->with('error', 'You do not have permission to view this.');
    }

    $request->validate([
        'new_students' => 'nullable|array',
        'existing_students' => 'nullable|array',
    ]);

    $yearId = Year::orderBy('year', 'desc')->first()->id;
    $year = Year::find($yearId)->year;
    $title = 'BPS Medical Terminology Upload Summary';

    $createdStudents = [];
    $enrolledExistingStudents = []; // This array holds students who were either existing and newly enrolled, or were new and just processed.
    $errors = new MessageBag();

    $schoolMap = [
        5 => 'CHS',
        6 => 'BHS',
        8 => 'LHS',
    ];

    // Process new students (or students whose data came from the 'new_students' array in preview)
    foreach ($request->new_students ?? [] as $rowData) {
        $rowData['ssid'] = (int)$rowData['ssid'];
        $rowData['phone'] = $rowData['phone'] ? (string)$rowData['phone'] : null;
        $section = Section::findOrFail($rowData['section_id']);

        // Find or create the student
        $student = Student::firstOrCreate(
            ['email' => $rowData['email']], // Use email for firstOrCreate, then check SSID
            [
                'first_name' => $rowData['first_name'],
                'last_name' => $rowData['last_name'],
                'school_id' => (int)$rowData['school_id'],
                'dob' => $rowData['dob'],
                'phone' => $rowData['phone'],
                'grade' => (int)$rowData['grade'],
                'gender' => $rowData['gender'],
                'ssid' => $rowData['ssid'], // Add SSID here if it's new
                'iep' => (bool)$rowData['iep'],
                'has_504' => (bool)$rowData['has_504'],
                'english_learner' => (bool)$rowData['english_learner'],
                'notes' => $rowData['notes'] ?? null,
            ]
        );

        // If a student with the email exists but SSID doesn't match, update SSID if null
        if ($student->wasRecentlyCreated === false && empty($student->ssid) && !empty($rowData['ssid'])) {
            $student->ssid = $rowData['ssid'];
            $student->save();
        }

        $sectionName = $section->course->name . ' (' . $year . ' ' . ($section->term == 'F' ? 'Fall' : 'Spring') . ') - ' . $section->teacher->first_name . ' ' . $section->teacher->last_name;
        $schoolName = $schoolMap[(int)$rowData['school_id']] ?? 'Unknown';

        // Create enrollment if it doesn't already exist for this student and section in the current year
        if (!$student->enrollments()->where('section_id', $section->id)->where('year_id', $yearId)->exists()) {
            Enrollment::create([
                'year_id' => $yearId,
                'student_id' => $student->id,
                'section_id' => $section->id,
                'course_id' => $section->course_id,
                'status' => 'A',
                'dual_credit' => false,
                'lms_updated' => false,
                'ps_updated' => false,
            ]);
            // If the student was created, add to createdStudents
            if ($student->wasRecentlyCreated) {
                $createdStudents[] = [
                    'id' => $student->id,
                    'name' => "{$rowData['first_name']} {$rowData['last_name']}",
                    'school' => $schoolName,
                    'section' => $sectionName,
                ];
            } else { // If student existed and was newly enrolled in this section
                $enrolledExistingStudents[] = [
                    'id' => $student->id,
                    'name' => "{$rowData['first_name']} {$rowData['last_name']}",
                    'school' => $schoolName,
                    'section' => $sectionName,
                ];
            }
        } else {
             // If student existed and was already enrolled in this section
            $enrolledExistingStudents[] = [
                'id' => $student->id,
                'name' => "{$rowData['first_name']} {$rowData['last_name']}",
                'school' => $schoolName,
                'section' => $sectionName,
            ];
        }
    }

    // Process students whose data came from the 'existing_students' array in preview
    foreach ($request->existing_students ?? [] as $rowData) {
        $rowData['ssid'] = (int)$rowData['ssid'];
        $rowData['phone'] = $rowData['phone'] ? (string)$rowData['phone'] : null;
        $section = Section::findOrFail($rowData['section_id']);
        $student = Student::where('email', $rowData['email'])->orWhere('ssid', $rowData['ssid'])->first();

        if ($student) {
            $sectionName = $section->course->name . ' (' . $year . ' ' . ($section->term == 'F' ? 'Fall' : 'Spring') . ') - ' . $section->teacher->first_name . ' ' . $section->teacher->last_name;
            $schoolName = $schoolMap[(int)$rowData['school_id']] ?? 'Unknown';

            // Create enrollment if it doesn't already exist
            if (!$student->enrollments()->where('section_id', $section->id)->where('year_id', $yearId)->exists()) {
                Enrollment::create([
                    'year_id' => $yearId,
                    'student_id' => $student->id,
                    'section_id' => $section->id,
                    'course_id' => $section->course_id,
                    'status' => 'A',
                    'dual_credit' => false,
                    'lms_updated' => false,
                    'ps_updated' => false,
                ]);
                $enrolledExistingStudents[] = [
                    'id' => $student->id,
                    'name' => "{$rowData['first_name']} {$rowData['last_name']}",
                    'school' => $schoolName,
                    'section' => $sectionName,
                ];
            } else {
                $enrolledExistingStudents[] = [ // Already enrolled
                    'id' => $student->id,
                    'name' => "{$rowData['first_name']} {$rowData['last_name']}",
                    'school' => $schoolName,
                    'section' => $sectionName,
                ];
            }
        } else {
            $errors->add('csv_error', "Student {$rowData['first_name']} {$rowData['last_name']} not found for enrollment.");
        }
    }

    // Pass 'enrolledExistingStudents' as 'existingStudents' to match the blade template
    return view('admin.enrollments.bps-medterm-upload-summary', compact('createdStudents', 'errors', 'title'))
           ->with('existingStudents', $enrolledExistingStudents); // Changed this line
}
    public function BPSMedTermStudentUploadForm()
    {
        if (auth()->user()->id != 1) {
            return redirect()->route('dashboard')->with('error', 'You do not have permission to view this.');
        }

        $yearId = Year::orderBy('year', 'desc')->first()->id;
        $year = Year::find($yearId)->year;

        $dropdowns = [
            'Fall BHS' => Section::where('sections.year_id', $yearId)
                ->where('sections.is_cancelled', false)
                ->whereRaw('restricted_school_ids IS NOT NULL AND JSON_CONTAINS(restricted_school_ids, ?)', ['"5"'])
                ->where('sections.term', 'F')
                ->join('courses', 'sections.course_id', '=', 'courses.id')
                ->join('teachers', 'sections.teacher_id', '=', 'teachers.id')
                ->select('sections.id', \DB::raw("CONCAT(courses.name, ' - ', teachers.first_name, ' ', teachers.last_name, ' (', ?, ' Fall)') as course_name"))
                ->addBinding($year, 'select')
                ->orderBy('courses.name')
                ->pluck('course_name', 'sections.id'),
            'Spring BHS' => Section::where('sections.year_id', $yearId)
                ->where('sections.is_cancelled', false)
                ->whereRaw('restricted_school_ids IS NOT NULL AND JSON_CONTAINS(restricted_school_ids, ?)', ['"5"'])
                ->where('sections.term', 'S')
                ->join('courses', 'sections.course_id', '=', 'courses.id')
                ->join('teachers', 'sections.teacher_id', '=', 'teachers.id')
                ->select('sections.id', \DB::raw("CONCAT(courses.name, ' - ', teachers.first_name, ' ', teachers.last_name, ' (', ?, ' Spring)') as course_name"))
                ->addBinding($year, 'select')
                ->orderBy('courses.name')
                ->pluck('course_name', 'sections.id'),
            'Fall CHS' => Section::where('sections.year_id', $yearId)
                ->where('sections.is_cancelled', false)
                ->whereRaw('restricted_school_ids IS NOT NULL AND JSON_CONTAINS(restricted_school_ids, ?)', ['"6"'])
                ->where('sections.term', 'F')
                ->join('courses', 'sections.course_id', '=', 'courses.id')
                ->join('teachers', 'sections.teacher_id', '=', 'teachers.id')
                ->select('sections.id', \DB::raw("CONCAT(courses.name, ' - ', teachers.first_name, ' ', teachers.last_name, ' (', ?, ' Fall)') as course_name"))
                ->addBinding($year, 'select')
                ->orderBy('courses.name')
                ->pluck('course_name', 'sections.id'),
            'Spring CHS' => Section::where('sections.year_id', $yearId)
                ->where('sections.is_cancelled', false)
                ->whereRaw('restricted_school_ids IS NOT NULL AND JSON_CONTAINS(restricted_school_ids, ?)', ['"6"'])
                ->where('sections.term', 'S')
                ->join('courses', 'sections.course_id', '=', 'courses.id')
                ->join('teachers', 'sections.teacher_id', '=', 'teachers.id')
                ->select('sections.id', \DB::raw("CONCAT(courses.name, ' - ', teachers.first_name, ' ', teachers.last_name, ' (', ?, ' Spring)') as course_name"))
                ->addBinding($year, 'select')
                ->orderBy('courses.name')
                ->pluck('course_name', 'sections.id'),
            'Fall LHS' => Section::where('sections.year_id', $yearId)
                ->where('sections.is_cancelled', false)
                ->whereRaw('restricted_school_ids IS NOT NULL AND JSON_CONTAINS(restricted_school_ids, ?)', ['"8"'])
                ->where('sections.term', 'F')
                ->join('courses', 'sections.course_id', '=', 'courses.id')
                ->join('teachers', 'sections.teacher_id', '=', 'teachers.id')
                ->select('sections.id', \DB::raw("CONCAT(courses.name, ' - ', teachers.first_name, ' ', teachers.last_name, ' (', ?, ' Fall)') as course_name"))
                ->addBinding($year, 'select')
                ->orderBy('courses.name')
                ->pluck('course_name', 'sections.id'),
            'Spring LHS' => Section::where('sections.year_id', $yearId)
                ->where('sections.is_cancelled', false)
                ->whereRaw('restricted_school_ids IS NOT NULL AND JSON_CONTAINS(restricted_school_ids, ?)', ['"8"'])
                ->where('sections.term', 'S')
                ->join('courses', 'sections.course_id', '=', 'courses.id')
                ->join('teachers', 'sections.teacher_id', '=', 'teachers.id')
                ->select('sections.id', \DB::raw("CONCAT(courses.name, ' - ', teachers.first_name, ' ', teachers.last_name, ' (', ?, ' Spring)') as course_name"))
                ->addBinding($year, 'select')
                ->orderBy('courses.name')
                ->pluck('course_name', 'sections.id'),
        ];

        return view('admin.enrollments.bps-medterm-upload', compact('dropdowns', 'yearId'))->with('title', 'Upload BPS Medical Terminology Students');
    }
}
