<?php

namespace App\Http\Controllers;

use App\Models\Report;
use App\Models\Enrollment;
use App\Models\Section;
use App\Models\Student;
use App\Models\Year;
use App\Models\School;
use App\Models\Provider;
use App\Models\Department;
use App\Models\Contact;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;
use Illuminate\Support\Facades\Mail;
use App\Mail\SchoolEnrollmentReport;

class ReportController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        if (auth()->user()->role_id != 1) {
            return redirect(route('dashboard'))->with('error','You do not have permission to view this.');
        }
        return view('admin.reports.index')
            ->with('title', 'Reports');
    }

    public function enrollmentsBySectionBySchool()
    {
        if (auth()->user()->role_id != 1) {
            return redirect(route('dashboard'))->with('error','You do not have permission to view this.');
        }
        return $this->enrollmentsBySectionBySchoolGetData(Year::orderBy('year', 'desc')->first()->id, 0,0,0,0,'A', 'A', 0);
    }

    public function enrollmentsBySectionBySchoolFiltered(Request $request)
    {
        if (auth()->user()->role_id != 1) {
            return redirect(route('dashboard'))->with('error','You do not have permission to view this.');
        }
        return $this->enrollmentsBySectionBySchoolGetData($request->year_id, $request->school_id, $request->provider_id, $request->department_id, $request->section_id, $request->status, $request->term, $request->grade);
    }

    public function enrollmentsBySectionBySchoolGetData($yearId, $schoolId, $providerId, $departmentId, $sectionId, $status, $term, $grade)
    {
        if (auth()->user()->role_id != 1) {
            return redirect(route('dashboard'))->with('error','You do not have permission to view this.');
        }
        // Show all sections or sections for provider or specific section selected
        $sectionsData = Section::where('year_id', $yearId);


        if ($sectionId > 0) {
            $sectionsData = $sectionsData->where('id', $sectionId);
        }

        switch (auth()->user()->role_id) {
            case 1:
                // Super User can see all enrollments
                if ($providerId > 0) {
                    $sectionsData = $sectionsData->where("provider_id", $providerId);
                }
                if ($departmentId > 0) {
                    $sectionsData = $sectionsData->join('courses', 'sections.course_id', '=', 'courses.id')->join('departments', 'courses.department_id', '=', 'departments.id')->select('sections.*', 'courses.department_id')->where('courses.department_id', '=', $departmentId);
                }
                break;
            default:
                // Providers can see all enrollments for their provider
                $sectionsData = $sectionsData->where("year_id", $yearId)->where("provider_id", auth()->user()->provider_id);
                break;
        }

        /* Show all schools or 1) school specific for the user or 2) school selected in dropdown */
        if(auth()->user()->role_id == 3) {
            $schools = School::where('id', auth()->user()->school_id)->get();
            $schoolsData = School::where('id', auth()->user()->school_id)->get();
        } else {
            if ($schoolId > 0){
                $schoolsData = School::where('id', $schoolId)->get();
            } else {
                $schoolsData = School::orderBy('name', 'asc')->get();
            }
        }

        if ($departmentId > 0) {
            $departmentsFiltered = Department::where("id", $departmentId)->get();

        } else {
            $departmentsFiltered = Department::orderBy('name', 'asc')->get();
        }
        return view('admin.reports.enrollments-by-section-by-school')
            ->with('yearId', $yearId)
            ->with('schoolId', $schoolId)
            ->with('providerId', $providerId)
            ->with('departmentId', $departmentId)
            ->with('sectionId', $sectionId)
            ->with('status', $status)
            ->with('term', $term)
            ->with('grade', $grade)
            ->with('years', Year::orderBy('year', 'desc')->get())
            ->with('schoolsData', $schoolsData)
            ->with('schools', School::orderBy('name', 'asc')->get())
            ->with('providers', Provider::orderBy('name', 'asc')->get())
            ->with('departments', Department::orderBy('name', 'asc')->get())
            ->with('sections', Section::where('year_id', $yearId)->get()->sortBy('course.name'))
            ->with('sectionsData', $sectionsData->get())
            ->with('title', 'Enrollments By Section By School');
    }

    public function enrollmentsByDepartment()
    {
        if (auth()->user()->role_id != 1) {
            return redirect(route('dashboard'))->with('error','You do not have permission to view this.');
        }
        return $this->enrollmentsByDepartmentGetData(Year::orderBy('year', 'desc')->first()->id, 0,0,'ALL', 'A', 0);
    }

    public function enrollmentsByDepartmentFiltered(Request $request)
    {
        if (auth()->user()->role_id != 1) {
            return redirect(route('dashboard'))->with('error','You do not have permission to view this.');
        }
        return $this->enrollmentsByDepartmentGetData($request->year_id, $request->department_id, $request->provider_id, $request->status, $request->term, $request->grade);
    }

    public function enrollmentsByDepartmentGetData($yearId, $departmentId, $providerId, $status, $term, $grade)
    {
        if (auth()->user()->role_id != 1) {
            return redirect(route('dashboard'))->with('error','You do not have permission to view this.');
        }

        if ($departmentId > 0) {
            $departmentsFiltered = Department::where("id", $departmentId)->get();

        } else {
            $departmentsFiltered = Department::orderBy('name', 'asc')->get();
        }

        if ($providerId > 0) {
            $providersFiltered = Provider::where("id", $providerId)->get();
        } else {
            $providersFiltered = Provider::orderBy('name', 'asc')->get();
        }

        return view('admin.reports.enrollments-by-department')
            ->with('yearId', $yearId)
            ->with('departmentId', $departmentId)
            ->with('providerId', $providerId)
            ->with('status', $status)
            ->with('term', $term)
            ->with('grade', $grade)
            ->with('years', Year::orderBy('year', 'desc')->get())
            ->with('providersFiltered', $providersFiltered)
            ->with('departmentsFiltered', $departmentsFiltered)
            ->with('providers', Provider::orderBy('name', 'asc')->get())
            ->with('departments', Department::orderBy('name', 'asc')->get())
            ->with('title', 'Enrollments By Department');
    }

    public function schoolEnrollmentReport()
    {
        if (auth()->user()->role_id != 1) {
            return redirect(route('dashboard'))->with('error', 'You do not have permission to view this.');
        }

        $years = Year::orderBy('year', 'desc')->get();
        $providers = Provider::orderBy('name', 'asc')->get();
        $departments = Department::orderBy('name', 'asc')->get();
        $schools = School::orderBy('name', 'asc')->get();

        $reportData = $this->getSchoolEnrollmentData(null, null, null, null, null, 'A');

        return view('admin.reports.school-enrollment-report', compact('reportData', 'years', 'providers', 'departments', 'schools'))
            ->with('title', 'School Enrollment Report')
            ->with('status', 'A');
    }

    public function schoolEnrollmentReportFiltered(Request $request)
    {
        if (auth()->user()->role_id != 1) {
            return redirect(route('dashboard'))->with('error', 'You do not have permission to view this.');
        }

        $years = Year::orderBy('year', 'desc')->get();
        $providers = Provider::orderBy('name', 'asc')->get();
        $departments = Department::orderBy('name', 'asc')->get();
        $schools = School::orderBy('name', 'asc')->get();

        $reportData = $this->getSchoolEnrollmentData(
            $request->year_id,
            $request->provider_id,
            $request->department_id,
            $request->school_id,
            $request->grade,
            $request->status ?? 'A'
        );

        return view('admin.reports.school-enrollment-report', compact('reportData', 'years', 'providers', 'departments', 'schools'))
            ->with('title', 'School Enrollment Report')
            ->with('yearId', $request->year_id)
            ->with('providerId', $request->provider_id)
            ->with('departmentId', $request->department_id)
            ->with('schoolId', $request->school_id)
            ->with('grade', $request->grade)
            ->with('status', $request->status ?? 'A');
    }

    public function getSchoolEnrollmentData($yearId = null, $providerId = null, $departmentId = null, $schoolId = null, $grade = null, $status = 'A')
{
    $query = Enrollment::join('sections', 'enrollments.section_id', '=', 'sections.id')
                       ->join('years', 'enrollments.year_id', '=', 'years.id')
                       ->join('students', 'enrollments.student_id', '=', 'students.id')
                       ->join('schools', 'students.school_id', '=', 'schools.id')
                       ->select('schools.name as school_name', 'years.year', 
                                DB::raw('SUM(CASE WHEN sections.term = "F" THEN 1 ELSE 0 END) AS fall_count'),
                                DB::raw('SUM(CASE WHEN sections.term = "S" THEN 1 ELSE 0 END) AS spring_count'),
                                DB::raw('SUM(CASE WHEN sections.term = "F" THEN 1 ELSE 0 END) AS full_year_count'))
                       ->groupBy('schools.name', 'years.year')
                       ->orderBy('schools.name');  // Order by school name alphabetically
    
    // Apply filters if provided
    if ($yearId) {
        $query->where('years.id', $yearId);
    }
    if ($providerId) {
        $query->where('sections.provider_id', $providerId); // Provider ID is in the 'sections' table
    }
    if ($departmentId) {
        $query->where('enrollments.department_id', $departmentId);
    }
    if ($schoolId) {
        $query->where('students.school_id', $schoolId);
    }
    if ($grade) {
        $query->where('students.grade', $grade);
    }
    if ($status) {
        $query->where('enrollments.status', $status);
    }
    
    // Execute the query and return the results
    return $query->get();
}

    
    
    public function sendSchoolEnrollmentReports(Request $request)
    {
        if (auth()->user()->role_id != 1) {
            return redirect(route('dashboard'))->with('error', 'You do not have permission to perform this action.');
        }

        $reportData = $this->getSchoolEnrollmentData(
            $request->year_id,
            $request->provider_id,
            $request->department_id,
            $request->school_id,
            $request->grade,
            $request->status ?? 'A'
        );

        $schoolCount = 0;

        foreach ($reportData as $schoolId => $schoolData) {
            if (!empty($schoolData['enrollments'])) {
                $facilitators = Contact::where('school_id', $schoolId)
                                       ->where('is_facilitator', true)
                                       ->get();

                foreach ($facilitators as $facilitator) {
                    Mail::to($facilitator->email)->send(new SchoolEnrollmentReport($schoolData));
                }

                $schoolCount++;
            }
        }

        return redirect()->back()->with('success', 'Enrollment reports sent for ' . $schoolCount . ' schools based on the current filters.');
    }

    public function showSchoolEnrollmentSummary()
    {
        if (auth()->user()->role_id != 1) {
            return redirect(route('dashboard'))->with('error', 'You do not have permission to view this.');
        }
    
        // Get necessary data for the report
        $years = Year::orderBy('year', 'desc')->get();
        $providers = Provider::orderBy('name', 'asc')->get();
        $schools = School::orderBy('name', 'asc')->get();
    
        // Fetch the report data with no filters
        $reportData = $this->getSchoolEnrollmentData(null, null, null, null, null, 'A');
    
        // Return the view with the necessary data
        return view('admin.reports.school-enrollment-summary', compact('reportData', 'years', 'providers', 'schools'))
            ->with('title', 'School Enrollment Summary Report')
            ->with('status', 'A');
    }

    public function showSchoolEnrollmentSummaryFiltered(Request $request)
    {
        if (auth()->user()->role_id != 1) {
            return redirect(route('dashboard'))->with('error', 'You do not have permission to view this.');
        }
    
        // Get necessary data for the report
        $years = Year::orderBy('year', 'desc')->get();
        $providers = Provider::orderBy('name', 'asc')->get();
        $schools = School::orderBy('name', 'asc')->get();
    
   
        // Fetch the filtered report data based on the user's request
        $reportData = $this->getSchoolEnrollmentData(
            $request->year_id,
            $request->provider_id,
            NULL,
            $request->school_id,
            NULL,
            'A'
        );
    
        // Return the view with the necessary data, including the filters
        return view('admin.reports.school-enrollment-summary', compact('reportData', 'years', 'providers', 'schools'))
            ->with('title', 'School Enrollment Summary Report')
            ->with('yearId', $request->year_id)
            ->with('providerId', $request->provider_id)
            ->with('departmentId', $request->department_id)
            ->with('schoolId', $request->school_id)
            ->with('grade', $request->grade)
            ->with('status', $request->status ?? 'A');
    }
         
    
}
