<?php

namespace App\Http\Controllers;

use App\Models\Section;
use App\Models\Year;
use App\Models\Course;
use App\Models\Department;
use App\Models\Teacher;
use App\Models\Provider;
use Illuminate\Http\Request;
use App\Http\Requests\StoreSectionRequest;
use App\Http\Requests\UpdateSectionRequest;
use App\Traits\SectionPermissionTrait;

class SectionController extends Controller
{
    use SectionPermissionTrait;

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        if ($this->checkSectionPermissions('index') == false) {
            return redirect(route('dashboard'))->with('error', 'You do not have permission to view sections.');
        }
        if (auth()->user()->role_id == 4) {
            $yearId = Year::where('is_visible_to_teachers', true)->orderBy('year', 'desc')->first()->id;
        } else {
            $yearId = Year::orderBy('year', 'desc')->first()->id;
        }
        return $this->index2($yearId, 0, 0, 0, 'A', 'A', 'A', 0);
    }

    public function index2($yearId, $providerId, $departmentId, $teacherId, $term, $dualCredit, $availability, $schoolId = 0)
    {
        switch (auth()->user()->role_id) {
            case 2:
                // Role: Provider
                $teachers = Teacher::orderBy('last_name', 'asc')->get();
                $providers = Provider::where('id', auth()->user()->provider_id)->get();
                $providerId = auth()->user()->provider_id;
                $years = Year::orderBy('year', 'desc')->get();
                break;
            case 4:
                $providers = Provider::where('id', auth()->user()->provider_id)->get();
                $teachers = Teacher::where('id', auth()->user()->teacher_id)->get();
                $teacherId = Teacher::where('id', auth()->user()->teacher_id)->first()->id;
                $years = Year::where('is_visible_to_teachers', true)->orderBy('year', 'desc')->get();
                break;
            default:
                $teachers = Teacher::orderBy('last_name', 'asc')->get();
                $providers = Provider::orderBy('name', 'asc')->get();
                $years = Year::orderBy('year', 'desc')->get();
                break;
        }
        return view('admin.sections.index')
            ->with('yearid', $yearId)
            ->with('providerId', $providerId)
            ->with('teacherId', $teacherId)
            ->with('departmentId', $departmentId)
            ->with('term', $term)
            ->with('availability', $availability)
            ->with('dualCredit', $dualCredit)
            ->with('schoolId', $schoolId)
            ->with('years', $years)
            ->with('teachers', $teachers)
            ->with('providers', $providers)
            ->with('departments', Department::orderBy('name', 'asc')->get())
            ->with('schools', \App\Models\School::whereIn('id', [5, 6, 8])->orderBy('name', 'asc')->get())
            ->with('title', 'Sections')
            ->with('sections', Section::sectionSQLBuilder($yearId, $providerId, $departmentId, $teacherId, $term, $dualCredit, $availability, $schoolId));
    }

    public function filterSections(Request $request)
    {
        if ($this->checkSectionPermissions('index') == false) {
            return redirect(route('dashboard'))->with('error', 'You do not have permission to view sections.');
        }
        return $this->index2(
            $request->year_id,
            $request->provider_id,
            $request->department_id,
            $request->teacher_id,
            $request->term,
            $request->dual_credit,
            $request->availability,
            $request->school_id ?? 0
        );
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        if ($this->checkSectionPermissions('create') == false) {
            return redirect(route('dashboard'))->with('error', 'You do not have permission to create sections.');
        }
        return view('admin.sections.form')
            ->with('action', 'create')
            ->with('title', 'Create Section')
            ->with('courses', Course::orderBy('name')->get())
            ->with('providers', Provider::orderBy('name')->get())
            ->with('years', Year::orderBy('year')->get())
            ->with('teachers', Teacher::orderBy('last_name')->get());
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreSectionRequest $request)
    {
        if ($this->checkSectionPermissions('store') == false) {
            return redirect(route('dashboard'))->with('error', 'You do not have permission to store sections.');
        }
        $data = $request->all();
        $restrictedSchoolIds = $request->input('restricted_school_ids', []);
        $data['restricted_school_ids'] = empty($restrictedSchoolIds) ? null : $restrictedSchoolIds;

        // For new sections, no enrollments exist, so no validation needed for restricted_school_ids
        Section::create($data);
        return redirect(route('sections.index'))->with('success', 'Section created successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(Section $section)
    {
        if ($this->checkSectionPermissions('show') == false) {
            return redirect(route('dashboard'))->with('error', 'You do not have permission to view that section.');
        }

        return view('admin.sections.show', ['section' => $section])->with('title', 'View ' . $section->name);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Section $section)
    {
        if ($this->checkSectionPermissions('edit') == false) {
            return redirect(route('dashboard'))->with('error', 'You do not have permission to view that section.');
        }
        return view('admin.sections.form')
            ->with('action', 'edit')
            ->with('title', 'Edit ' . $section->course->name . ' ' . $section->year->year . ' ' . $section->termDesc())
            ->with('section', $section)
            ->with('courses', Course::orderBy('name')->get())
            ->with('providers', Provider::orderBy('name')->get())
            ->with('years', Year::orderBy('year')->get())
            ->with('teachers', Teacher::orderBy('last_name')->get());
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateSectionRequest $request, Section $section)
    {
        if ($this->checkSectionPermissions('update') == false) {
            return redirect(route('dashboard'))->with('error', 'You do not have permission to view that section.');
        }
        $data = $request->all();
        $restrictedSchoolIds = $request->input('restricted_school_ids', []);
        $data['restricted_school_ids'] = empty($restrictedSchoolIds) ? null : $restrictedSchoolIds;

        // Check if the section has enrollments from schools not in the new restricted_school_ids
        if (!empty($data['restricted_school_ids']) && $section->enrollments()->count() > 0) {
            $enrolledSchoolIds = $section->enrollments()
                ->join('students', 'enrollments.student_id', '=', 'students.id')
                ->pluck('students.school_id')
                ->unique()
                ->toArray();

            $invalidSchools = array_diff($enrolledSchoolIds, $data['restricted_school_ids']);
            if (!empty($invalidSchools)) {
                $schoolNames = \App\Models\School::whereIn('id', $invalidSchools)
                    ->pluck('name')
                    ->implode(', ');
                return redirect()->back()->with('error', "Cannot update restrictions. Students from {$schoolNames} are already enrolled.");
            }
        }

        $section->update($data);
        return redirect(route('sections.index'))->with('success', 'Section ' . $section->name . ' updated successfully!');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function delete(Section $section)
    {
        if ($this->checkSectionPermissions('delete') == false) {
            return redirect(route('dashboard'))->with('error', 'You do not have permission to view that section.');
        }

        if ($section->enrollments->count() > 0) {
            return redirect()->route('sections.index')->with('error', 'Cannot delete section. It has active enrollments.');
        }

        return view('admin.sections.delete')
            ->with('section', $section)
            ->with('title', 'Delete ' . $section->name);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Section $section)
    {
        if ($this->checkSectionPermissions('destroy') == false) {
            return redirect(route('dashboard'))->with('error', 'You do not have permission to view that section.');
        }

        if ($section->enrollments->count() > 0) {
            return redirect()->route('sections.index')->with('error', 'Cannot delete section. It has active enrollments.');
        }

        $section->delete();
        return redirect(route('sections.index'))->with('success', 'Section ' . $section->name . ' deleted successfully!');
    }

    /**
     * Display sections available for enrollment, filtered by school restrictions.
     */
    public function showSectionsForEnrollment($yearId, $providerId, $term)
    {
        $user = auth()->user();
        $schoolId = $user->school_id;

        $query = Section::where('year_id', $yearId)
            ->where('is_cancelled', false)
            ->when($providerId != 0, function ($query) use ($providerId) {
                return $query->where('provider_id', $providerId);
            })
            ->when($term != 'A', function ($query) use ($term) {
                return $query->where('term', $term);
            })
            ->where(function ($query) use ($schoolId) {
                $query->whereNull('restricted_school_ids')
                      ->orWhereJsonContains('restricted_school_ids', (int) $schoolId);
            })
            ->get()
            ->sortBy(['provider.name', 'course.name']);

        return view('admin.sections.sections-by-term', ['sections' => $query]);
    }
}