<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\DB;
use App\Traits\EnrollmentPermissionTrait;

class Enrollment extends Model
{
    use EnrollmentPermissionTrait;
    use HasFactory;

    protected $fillable = [
        'year_id',
        'student_id',
        'section_id',
        'course_id',
        'status',
        'lms_updated',
        'ps_updated',
        'dual_credit',
        'notes',
        'sent_welcome_letter',
        'sent_facilitator_copy',
    ];
    
    public function student(): BelongsTo
    {
        return $this->belongsTo(Student::class, 'student_id');
    }

    public function section(): BelongsTo
    {
        return $this->belongsTo(Section::class, 'section_id');
    }

    public function welcomeLetter(): BelongsTo
    {
        return $this->belongsTo(WelcomeLetter::class);
    }

    public function year(): BelongsTo
    {
        return $this->belongsTo(Year::class, 'year_id');
    }

    public function statusDesc()
    {
        switch($this->status){
            case('A'):
                return 'Approved';
                break;
            case('D'):
                return 'Denied';
                break;
            case('R'):
                return 'Dropped';
                break;
            default:
                return 'Waitlist';
        }
    }

    public function canViewOrDrop(){
        switch(auth()->user()->role_id){
            case 1:
                return true;
                break;
            case 2:
                if ($this->section->provider_id === auth()->user()->provider_id){
                    return true;
                } else {
                    return false;
                }
                break;
            case 3:
                if (auth()->user()->school_id == $this->student->school_id){
                    return true;
                } else {
                    return false;
                }
                break;
            default:
                return false;
        }
    }

    public static function enrollmentSQLBuilder($yearId, $schoolId, $providerId, $departmentId, $sectionId, $teacherId, $status, $term, $grade){
        $sql = DB::table("enrollments as e")
        ->join("students as stu", function($join){
            $join->on("stu.id", "=", "e.student_id");
        })
        ->join("schools as sch", function($join){
            $join->on("sch.id", "=", "stu.school_id");
        })
        ->join("sections as sec", function($join){
            $join->on("sec.id", "=", "e.section_id");
        })
        ->join("teachers as t", function($join){
            $join->on("t.id", "=", "sec.teacher_id");
        })
        ->join("courses as c", function($join){
            $join->on("c.id", "=", "sec.course_id");
        })
        ->join("departments as d", function($join){
            $join->on("d.id", "=", "c.department_id");
        })
        ->join("providers as p", function($join){
            $join->on("p.id", "=", "sec.provider_id");
        })
        ->select("e.*", "stu.id as studentid", "sch.id as schoolid", "sec.id as sectionid", "sec.enrollment_max", "d.id as departmentid", "p.id as providerid", "t.id as teacherId", "sec.is_cancelled");

        if ($yearId > 0) {
            $sql = $sql->where("e.year_id", "=", $yearId);
        }
        if ($schoolId > 0) {
            $sql = $sql->where("sch.id", "=", $schoolId);
        }
        if ($departmentId > 0) {
            $sql = $sql->where("d.id", "=", $departmentId);
        }
        if ($providerId > 0) {
            $sql = $sql->where("p.id", "=", $providerId);
        }
        if ($sectionId > 0) {
            $sql = $sql->where("sec.id", "=", $sectionId);
        }
        if ($teacherId > 0) {
            $sql = $sql->where("sec.teacher_id", "=", $teacherId);
        }
        if ($status != 'ALL') {
            $sql = $sql->where("e.status", "=", $status);
        }
        if ($term != 'A') {
            $sql = $sql->where("sec.term", "=", $term);
        }
        if ($grade > 0) {
            $sql = $sql->where("stu.grade", "=", $grade);
        }

        switch (auth()->user()->role_id) {
            case 1:
                // Super User can see all enrollments
                break;
            case 2:
                // Providers can see all enrollments for their provider
                $sql = $sql->where("p.id", "=", auth()->user()->provider_id);
                break;
            case 3:
                // School can see all enrollments for their school only
                $sql = $sql->where("sch.id", "=", auth()->user()->school_id);
                break;
            default:
                $sql = $sql->where("t.id", "=", auth()->user()->teacher_id);

        }
        //ddd($sql->toRawSql());
        return $sql->get();
    }
}
