<?php

namespace App\Services;

use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use App\Models\Section;
use App\Models\Year;
use App\Models\WelcomeLetter;

class EmailService
{
    private $uploadBasePath;

    public function __construct()
    {
        $this->uploadBasePath = public_path('uploads/');
    }

    public function sendWelcomeEmail($sectionId, $to, $teacherEmail, $teacherName, $subject, $body, $bccEmails)
    {
        $fromEmail = config('mail.from.address');
        $fromName = $teacherName . " via " . config('app.name');

        $emailBody = $body . "\n\n---\n" .
            "This email was sent on behalf of {$teacherName} ({$teacherEmail}). " .
            "You can reply directly to this email to reach the teacher.";

        Log::info("Attempting to send welcome email for section: $sectionId");

        try {
            Mail::send([], [], function ($message) use ($sectionId, $to, $fromEmail, $fromName, $teacherEmail, $subject, $emailBody, $bccEmails) {
                $message->to($to)
                        ->from($fromEmail, $fromName)
                        ->subject($subject)
                        ->html($emailBody)
                        ->replyTo($teacherEmail);

                if (!empty($bccEmails)) {
                    $bccArray = array_filter(array_map('trim', explode(';', $bccEmails)));
                    if (!empty($bccArray)) {
                        $message->bcc($bccArray);
                    }
                }

                $this->attachSchoolCalendar($message, $sectionId);
                $this->attachTravelConsentForm($message, $sectionId);
                $this->attachWelcomeLetterFiles($message, $sectionId);
            });

            Log::info("Welcome email sent successfully for section: $sectionId");
            return true;
        } catch (\Exception $e) {
            Log::error("Error sending welcome email for section $sectionId: " . $e->getMessage());
            return false;
        }
    }

    private function attachSchoolCalendar($message, $sectionId)
    {
        $section = Section::findOrFail($sectionId);
        $school_calendar_attachment = ($section->provider_id == 4)
            ? "2024-25-basc-calendar.pdf"
            : Year::find($section->year_id)->school_calendar_attachment;

        $calendarPath = $this->uploadBasePath . 'school-calendars/' . $school_calendar_attachment;

        if (file_exists($calendarPath)) {
            $message->attach($calendarPath, [
                'as' => $school_calendar_attachment,
                'mime' => 'application/pdf',
            ]);
            Log::info("School calendar attached: $calendarPath");
        } else {
            Log::warning("School calendar not found: $calendarPath");
        }
    }

    private function attachTravelConsentForm($message, $sectionId)
    {
        $section = Section::findOrFail($sectionId);
        if($section->provider_id != 1){
            return;
        }else {
            $travel_consent_form_attachment = "cractc-travel-consent-and-participation-permission-form.docx";

            $travelFormPath = $this->uploadBasePath . $travel_consent_form_attachment;

            if (file_exists($travelFormPath)) {
                $message->attach($travelFormPath, [
                    'as' => "CRACTC Travel Consent and Participation Permission Form.docx",
                    'mime' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                ]);
                Log::info("Travel Permission Form attached: $travelFormPath");
            } else {
                Log::warning("Travel Permission Form not found: $travelFormPath");
            }
        }

    }

    private function attachWelcomeLetterFiles($message, $sectionId)
    {
        $welcomeLetter = WelcomeLetter::where('section_id', $sectionId)->first();
        $fileFields = ['file1', 'file2', 'file3'];
        foreach ($fileFields as $field) {
            if (isset($welcomeLetter->$field)) {
                $filePath = $this->uploadBasePath . 'welcome-letter-files/' . $welcomeLetter->$field;
                $originalFileName = $welcomeLetter->$field;
                $userProvidedFileName = $welcomeLetter->{$field . '_title'} ?? $originalFileName;

                if (file_exists($filePath)) {
                    $fileExtension = pathinfo($originalFileName, PATHINFO_EXTENSION);
                    $mimeType = $this->getMimeType($filePath);

                    // Ensure the user-provided file name has the correct extension
                    $fileNameWithExtension = pathinfo($userProvidedFileName, PATHINFO_EXTENSION)
                        ? $userProvidedFileName
                        : $userProvidedFileName . '.' . $fileExtension;

                    $message->attach($filePath, [
                        'as' => $fileNameWithExtension,
                        'mime' => $mimeType,
                    ]);
                    Log::info("Welcome letter file attached: $filePath as $fileNameWithExtension");
                } else {
                    Log::warning("Welcome letter file not found: $filePath");
                }
            }
        }
    }

    private function getMimeType($filePath)
    {
        if (!file_exists($filePath)) {
            Log::error("File not found: $filePath");
            return false;
        }

        if (!is_readable($filePath)) {
            Log::error("File not readable: $filePath");
            return false;
        }

        $extension = strtolower(pathinfo($filePath, PATHINFO_EXTENSION));

        $mimeTypes = [
            // Microsoft Office
            'doc'  => 'application/msword',
            'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'xls'  => 'application/vnd.ms-excel',
            'xlsx' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'ppt'  => 'application/vnd.ms-powerpoint',
            'pptx' => 'application/vnd.openxmlformats-officedocument.presentationml.presentation',

            // PDF
            'pdf'  => 'application/pdf',

            // Images
            'jpg'  => 'image/jpeg',
            'jpeg' => 'image/jpeg',
            'png'  => 'image/png',
            'gif'  => 'image/gif',
            'bmp'  => 'image/bmp',
            'tiff' => 'image/tiff',
            'svg'  => 'image/svg+xml',

            // Text
            'txt'  => 'text/plain',
            'csv'  => 'text/csv',
            'html' => 'text/html',
            'htm'  => 'text/html',
            'xml'  => 'application/xml',
            'json' => 'application/json',

            // Compressed files
            'zip'  => 'application/zip',
            'rar'  => 'application/x-rar-compressed',
            '7z'   => 'application/x-7z-compressed',

            // Audio
            'mp3'  => 'audio/mpeg',
            'wav'  => 'audio/wav',

            // Video
            'mp4'  => 'video/mp4',
            'avi'  => 'video/x-msvideo',
        ];

        if (isset($mimeTypes[$extension])) {
            return $mimeTypes[$extension];
        }

        // Fallback to mime_content_type if extension not in our list
        try {
            return mime_content_type($filePath);
        } catch (\Exception $e) {
            Log::error("Error determining mime type: " . $e->getMessage());
            return 'application/octet-stream'; // Default to binary data if we can't determine the type
        }
    }
}
